
//  *****************************************************************  //
//
//  PSim:  discrete-event simulator, based on user-level threads 
//  *****************************************************************  //

//  Elevator simulation.  People arrive to the elevator waiting area at
//  each floor at random times, and choose random floors as their
//  destinations.  The elevator will continue in its current direction
//  as long as a current rider has a destination in that direction, or
//  if someone has summoned it from a floor in that direction.  The
//  elevator remains still at its current floor if no one is waiting for
//  it.
//
//  Since this program is meant as an illustration of PSim, in the
//  interests of simplicity we are assuming here that, once the elevator
//  starts toward a given floor, it will not stop at any intermediate
//  floors at which requests come in during this trip.

//  usage:  elevator debug_flag time_limit num_floors mean_interarrival

#include <string>
#include <cstdlib>
#include <math.h>

#include <PSimGlobals.h>

#define MAX_FLOORS 10

float MeanInterarrival;  // mean interrival time 
int NFloors;  // number of floors

float TotWait;  // total passenger elevator wait so far
int NRiders;  // total number of passenger trips done so far

// data for waiting passengers
int NW[MAX_FLOORS];  // array of numbers waiting at each floor
int TotNW;  // sum of all NW[I]
float SumArrv[MAX_FLOORS];  // sums of arrival times
int WaitDest[MAX_FLOORS][MAX_FLOORS];  // counts of destinations
int SummoningSrc,SummoningDst;

// data for passengers in elevator
int ElvDest[MAX_FLOORS],  // counts of destinations
    ElvPass = 0;  // current number of passengers

// current position, etc. of elevator
int CurrFloor = 0,  // current location
    CurrDirection = 1,  // current direction, up (+1) or down (-1)
    CurrDest;  // current destination 

PTHSAppThread *ArrivalsThread;
PTHSAppThread *ElevatorThread;

// could have had a separate arrivals thread for each floor, obviating
// the need for the following random number generation, but fewer
// threads makes for easier debugging

void *Arrivals(void *)

{  int Job=0;
   float FT=0.0;
   PTHSJob *NewArrival;
   int Src,Dst;

   for (Job = 0; ; Job++)  {
      NewArrival = new PTHSJob;
      FT = PTHSSimTime + PTHSExpon(MeanInterarrival);
      strcpy(NewArrival->PTHSEvntName,"person arrival");
      NewArrival->PTHSJobNum = Job;
      NewArrival->PTHSEvntTime = FT;
      Src = (int) floor(NFloors*PTHSRnd());
      // destination must be different from source
      do  {
         Dst = (int) floor(NFloors*PTHSRnd());
      } while (Dst == Src);
      NewArrival->PTHSHold();
      WaitDest[Src][Dst]++;
      NW[Src]++;
      SumArrv[Src] += PTHSSimTime;
      if (TotNW++ == 0 && ElvPass == 0) { // ElevatorThread needs to be waked
         SummoningSrc = Src;
         SummoningDst = Dst;
         ElevatorThread->PTHSSignalAppThread();
      }
   }
}

void Update1()

{  if (SummoningSrc < CurrFloor)  {
      CurrDirection = -1;
      CurrDest = SummoningSrc;
      return;
   }
   if (SummoningSrc > CurrFloor)  {
      CurrDirection = 1;
      CurrDest = SummoningSrc;
      return;
   }
   // summoning source is current floor, so direction will be determined
   // by summoning destination; also, since we are already there, must
   // update the various totals NOW; we will take on one passenger from
   // floor SummoningSrc, bound for SummoningDst
   if (SummoningDst < CurrFloor) CurrDirection = -1;
   else CurrDirection = 1;
   CurrDest = SummoningDst;
   NRiders++;
   ElvDest[SummoningDst]++;
   ElvPass++;
   WaitDest[SummoningSrc][SummoningDst] = 0;
   TotNW--;
   TotWait += PTHSSimTime - SumArrv[SummoningSrc];
   NW[SummoningSrc]--;
   SumArrv[SummoningSrc] = 0.0;
}

void DropOffAndPickUp()

{  
   // current passengers alight
   ElvPass -= ElvDest[CurrDest];
   ElvDest[CurrDest] = 0;
   // new passengers move from waiting area into elevator
   NRiders += NW[CurrDest];
   for (int J = 0; J < NFloors; J++)  {
      ElvDest[J] += WaitDest[CurrDest][J];
      ElvPass += WaitDest[CurrDest][J];
      WaitDest[CurrDest][J] = 0;
   }
   TotNW -= NW[CurrDest];
   NW[CurrDest] = 0;
   SumArrv[CurrDest] = 0.0;
}

void Update2()  

{  int NF1 = NFloors - 1;

   if (CurrFloor == 0 || CurrFloor == NFloors-1) 
      CurrDirection = - CurrDirection;
   for (int I = CurrFloor+CurrDirection; 0 <= I <= NF1; I += CurrDirection)  {
      if (WaitDest[I] > 0 || ElvDest[I] > 0)  {
         CurrDest = I;
         return;
      }
   }
}

void *Elevator(void *)

{  float FT=0.0;  // finish time for current trip
   PTHSJob ElvJob;

   strcpy(ElvJob.PTHSEvntName,"elevator arrival");

   while (1)  {
      // quiet period, no passengers and no summons
      ElevatorThread->PTHSWaitAppThread();
      // active period
      Update1();
      while (1)  {
         FT = PTHSSimTime + 1.0*abs(CurrFloor-CurrDest);
         ElvJob.PTHSEvntTime = FT;
         ElvJob.PTHSHold();
         CurrFloor = CurrDest;
         DropOffAndPickUp();
         // now rest or move?
         if (TotNW == 0 && ElvPass == 0) break;  // start another quiet period
         Update2();
      }
   }
}

void OurInit(int Argc, char **Argv)

{  sscanf(Argv[3],"%d",&NFloors);
   sscanf(Argv[4],"%f",&MeanInterarrival);
   TotWait = 0.0;
   PTHSNAT = 2;
   ArrivalsThread = new PTHSAppThread(&Arrivals,NULL);
   ElevatorThread  = new PTHSAppThread(&Elevator,NULL);
   TotNW = 0;
   for (int I = 0; I < MAX_FLOORS; I++)  {
      NW[I] = 0;
      SumArrv[I] = 0.0;
      for (int J = 0; J < MAX_FLOORS; J++)  {
         WaitDest[I][J] = 0;
      }
   }
}

int main(int argc, char **argv)

{  PTHSInit(argc,argv);  
   OurInit(argc,argv);
   PTHSEndSm.PTHSWaitDone();
   printf("warning:  this program not thoroughly tested\n");
   printf("mean wait for %d jobs = %f\n",NRiders,TotWait/NRiders);
   exit(1);
}

