#include <stdio.h>
#include <ctype.h>
#include "metastr.h"
#define EQU 1
#define END 2
#define STOP 3
#define CONST 4
#define ORG 5
#define WRONG 6

extern int curloc;
extern int srclines;
extern INSTR_FORMAT format[20];
extern OPCODE_TABLE ot;
extern SVP symtable[500];
extern int nsymtable;

/***************************************************************/
void pass_one(fp)
/***************************************************************/
FILE *fp;
/*
  first pass of two-pass assembler.  Gathers label values,
  checks for syntax
                                                               */
{
  char line[81], label[20], ae[60], msg[80];
  int nch, itype,builtin,foundlabel,value;
  void proc_label(), proc_opcode(), proc_remainder(), proc_equ(),
           tosymtable();

getline:
  if(feof(fp)) return;
  fgets(line,81,fp);convert_lc(line);++srclines;
/*
   comment?
                */
  if(!strncmp(line,";",1)) goto getline;
  nch = 0;
  proc_label(line,&nch,label,&foundlabel);
  proc_opcode(line,&nch ,&itype,&builtin);
/*
   special processing for fixed assembler opcode (STOP) and
   directives (END, EQU, ORG, CONST)
                                                           */
  if(builtin)
   switch(builtin)
    {case EQU:  if(foundlabel)
                    proc_equ(line,&nch,label);
                else
                    error(" must specify a label when using 'equ' ");
                goto getline;
     case END:  return;
     case STOP: if(foundlabel) tosymtable(label,curloc);
                ++curloc;
	        goto getline;
     case CONST:if(foundlabel) tosymtable(label,curloc);
	        if(!get_ae(line,&nch,ae))
                  error(" illegal operand for 'const' ");
                else
	          ++curloc;
	        goto getline;
     case ORG:  if(!get_ae(line,&nch,ae))
	         error(" illegal operand for 'org' ");
                else
	         {value = evalae(ae);
	          if(value < curloc)
                   {sprintf(msg,
		     " org value of %d less than current location counter %d",
			value,curloc);
		    warning(msg);
	            warning(" may overwrite some previous code");
                   }
		  curloc = value;
		  if(foundlabel) tosymtable(label,curloc);
                 }
     case WRONG:goto getline;
    }
  else
/*
  here to process a valid opcode
                                                           */
   {if(foundlabel) tosymtable(label,curloc);
    proc_remainder(line,&nch,itype);
/*
   update memory location counter
                                                           */
    curloc += format[itype].nwords;
    goto getline;
   }
}
/***************************************************************/
void proc_label(line,nch,label,found)
/***************************************************************/
char *line;
int *nch;
char *label;
int *found;
/*
  checks to see if there is a label on this line.  If so
  returns it and sets found to 1
                                                               */
{
  char str[80];
  int status;

  if(!span_white(line,nch))
   {error("blank lines illegal");
    *found = 0;
    return;
   }
  if(*nch == 0)
   {if(an_str(line,nch,label)<= 0)
       {error("nonalphanumeric label was encountered");
        *found = 0;
        return;
       }
    *found = 1;
    if(strlen(label) > 7)
      {sprintf(str, "truncating %s to 7 characters", label);
       label[8] = '\0';
      }
    return;
   }
  *found = 0;
  return;
}
/***************************************************************/
void proc_opcode(line,nch,itype,builtin)
/***************************************************************/
char *line;
int *nch,*itype;
int *builtin;
{
  char token[50], str[81];
  int status, j;

  status = an_str(line,nch,token);
  if(status == 0)
   {error("illegal leading character in opcode");
    *builtin = WRONG;
    return;
   }
  if(status == -1)
   {error("opcode missing");
    *builtin = WRONG;
    return;
   }
/*
  found a token
                 */
  if(!strncmp(token,"equ",3))
      {*builtin = EQU;
       return;
      }
  if(!strncmp(token, "end",3))
      {*builtin = END;
       return;
      }
  if(!strncmp(token, "stop",4))
      {*builtin = STOP;
       return;
      }
  if(!strncmp(token, "const",5))
      {*builtin = CONST;
	return;
      }
  if(!strncmp(token, "org",3))
      {*builtin = ORG;
	return;
      }
/*
  token is not a builtin assembler instruction or pseduo op.
  check the opcode name table
                                                             */
  for(j=0;j<ot.n;++j)
    if(!strcmp(token,ot.op[j].sym))
      {/*
             make sure we're matching with an opcode
                                                             */
        if(ot.op[j].format >= 0)
            {*builtin = 0;
             *itype = ot.op[j].format;
            }
        else
            {*builtin = WRONG;
             sprintf(str,"%s is not a legal opcode",token);
             error(str);
            }
        return;
      }
  sprintf(str,"%s is not a legal opcode",token);
  error(str);
  *builtin = WRONG;
  return;
}
#define ILLEGAL -2
#define NOPUNCT 0
#define HITEOL -1
#define LEGAL 1
/***************************************************************/
void proc_remainder(line,nch,itype)
/***************************************************************/
char *line;
int *nch;
int itype;
/*
   processes all fields after the opcode
                                                               */
{
  int j, nf, index,ktype,status, value;
  char str[50],pstr[2],msg[81];
  FIELD *p;

  nf = format[itype].nfields;
  p = format[itype].f;
  p = p->next;           /* get past the opcode field         */
  --nf;                  /* not processing the opcode field   */
  for(j=0;j<nf;++j)
   {if(p->reg || p->constant || p->key)
      {if(get_rlk(line,nch,p,str,&index,&ktype) <= 0)
          {sprintf(msg,"illegal %s field",p->fname);
           error(msg);
           return;
          }
      }
    if(p->punct)
      {status = get_lpunct(line,nch,pstr,&value);
       if(status == ILLEGAL)
           {sprintf(msg,"illegal %s field",p->fname);
            error(msg);
            return;
           }
      }
   if(j != nf-1)
       p = p->next;
   }
  return;
}
/***************************************************************/
void tosymtable(sym,value)
/***************************************************************/
/*
   if value > -1, puts symbol and its value into the symbol
   table.  Checks to be sure that symbol wasn't defined 
   previously.  If value == -1, value isn't known yet.  If
   symbol isn't in the table yet, put it in.
                                                               */
{
  int j;
  char str[80];

  if(value < 0)
    {for(j=0;j<nsymtable;++j)
       if(!strcmp(symtable[j].sym,sym))return;
     if(nsymtable == 500)
       fatal_error("symbol table overflow.  contact programmer");
     strcpy(symtable[nsymtable].sym,sym);
     symtable[nsymtable].value = value;
     ++nsymtable;
     return;
    }
  if(value >= 0)
    {for(j=0;j<nsymtable;++j)
       if(!strcmp(symtable[j].sym,sym))
          {if(symtable[j].value < 0)
               symtable[j].value = value;
           else
             {sprintf(str,"multiple definition of label %s",sym);
              error(str);
             }
           return;
          }
     if(nsymtable == 500)
       fatal_error("symbol table overflow.  contact programmer");
     strcpy(symtable[nsymtable].sym,sym);
     symtable[nsymtable].value = value;
     ++nsymtable;
     return;
    } 
}   
/***************************************************************/
void proc_equ(line,nch,label)
/***************************************************************/
char *line;
int *nch;
char *label;
/*
   gets the value for the label
                                                               */
{
   int num;

   if(get_num(line,nch,&num) < 1)
     {error(" value in equ statement must be numeric");
      return;
     }
   tosymtable(label,num);
   return;
}
/***************************************************************/
int get_rlk(line,nch,p,ae,index,ktype)
/***************************************************************/
/*
  gets and checks any alphanumeric string (keyword, register)
  or arithmetic expression (constant)
                                                               */
char *line;
int *nch;
FIELD *p;
char *ae;
int *index;
int *ktype;
{
  if(!get_ae(line,nch,ae)) return 0;
  if(p->reg)
    {*index = isreg(ae); 
     if(*index >= 0)
       {*ktype = 1;
        return 1;
       }
    }
  if(p->key)
    {*index = iskey(ae);
     if(*index >= 0)
       {*ktype = 2;
        return 1;
       }
    }
  if(p->constant)
    {*index = analae(ae);
     if(*index >= 0)
       {*ktype = 3;
        return 1;
       }
    }
  return 0;
} 
/***************************************************************/
int get_lpunct(line,nch,pstr,value)
/***************************************************************/
/*
    find character in a punctuation field.  returns
           LEGAL  -- found user-defined symbol
           NOPUNCT - no punctuation.  Assuming "null"
           HITEOL -- no punctuation.  Assuming "null"
          ILLEGAL -- illegal punct or no "null" defined
                                                               */
char *line;
int *nch;
char *pstr;
int *value;
{
  int nch2, status;
  char msg[81];

  if(!span_comma(line,nch))
/*
   nothing here.  assume we have null for punctuation
                                                              */
     {*value = iskey("null");
      if(*value < 0)
        {error("missing punctuation");
         return ILLEGAL;
        }
      return NOPUNCT;
     }
  nch2 = *nch;
  status = get_punct(line,&nch2,pstr);
  switch(status)
    {case ILLEGAL: sprintf(msg,"illegal character %s encountered",pstr);
                   error(msg);
                   return ILLEGAL;
     case NOPUNCT:
     case HITEOL:  *value = iskey("null");
                   if(*value < 0)
                      {error("missing punctuation");
                       return ILLEGAL;
                      }
                   return NOPUNCT;
    case LEGAL:    *value = isdefpunct(pstr);
                   *nch = nch2;
                   return LEGAL;
    }
}
/***************************************************************/
int get_punct(line,nch,pstr)
/***************************************************************/
/*
   gets and classifies the next character from the
   input string
                                                               */
char *line;
int *nch;
char *pstr;
{
  pstr[0] = line[*nch];
  pstr[1] = '\0';
  if(isdefpunct(pstr) >= 0)
     {++(*nch);
      return LEGAL;
     }
  if(isalnum(line[*nch])) return NOPUNCT;
  if(ispunct(line[*nch])) return ILLEGAL;
  return NOPUNCT;
}
