#ifndef lint
static char     sccsid[] = "@(#)linda_util.c";
static char     Creator[] = "C 1991 G. Sxoinas csi.forth.gr \n";
static char     Mail[] = "sxoinas@ariadne.bitnet or sxoinas@csi.forth.gr";
#endif
/*
 * This file is part of the POSYBL (PrOgramming SYstem for distriButed
 * appLications, a free Linda implementation for Unix Networks) and contains
 * the various data copying and inspection routines used throughout the system.
 * 
 * There are no restrictions on this code; however, if you make any changes, I
 * request that you document them so that I do not get credit or blame for your
 * modifications.
 * 
 * Written by Ioannis Schoinas (sxoinas@csd.uch.gr or sxoinas@csi.forth.gr),
 * Computer Science Department, University of Crete, Heraclion, Crete, Greece.
 * 
 * Note: Linda is a trademark of SCA, Inc.
 * 
 */
#include <stdio.h>

#include "linda_defs.h"

void
copystring(d, s)
  char           *s, *d;
{
  strcpy(d, s);
}

void
copychars(d, s, l)
  register char  *s, *d;
  register int    l;
{
  memcpy(d, s, l);
}

void
copyshorts(d, s, l)
  register short *s, *d;
  register int    l;
{
  while (--l >= 0)
    *d++ = *s++;
}

void
copyints(d, s, l)
  register int   *s, *d;
  register int    l;
{
  while (--l >= 0)
    *d++ = *s++;
}

void
copyfloats(d, s, l)
  register float *s, *d;
  register int    l;
{
  while (--l >= 0)
    *d++ = *s++;
}

void
copydoubles(d, s, l)
  register double *s, *d;
  register int    l;
{
  while (--l >= 0)
    *d++ = *s++;
}

/*
 * The following routine is just a dirty trick, that is based on the fact that
 * in modern workstations is faster to copy an array of bytes by accessing them
 * as integers than as bytes. Of course, in order to do so certain alignment
 * constrains must be fulfilied. Since malloc returns word aligned areas and
 * since all the data used by the tmanager are padded to word boundaries the
 * code below results in faster copying of byte arrays. Unfortunately this is
 * not the case for shorts, since their implementation by C is just masked
 * operation on integers. And it probably won't pay off to try access them as
 * integers in any case by copying the first few as bytes and the rest,
 * properly alignd now, as integers, since their expected size is small and the
 * overhead per call would just slow things down.
 */

void
copybytes(d, s, l)
  register char  *d, *s;
  register int    l;
{
  if ((((int) d) & 0x00000003) == 0 &&
      (((int) s) & 0x00000003) == 0 && (l & 0x00000003) == 0)
    copyints(d, s, l >> 2);
  else
    copychars(d, s, l);
}

int
field_bytes(field)		/* Returns the size of a field from the
				 * field_struct pointed by field */
  register FIELD *field;
{
  register int    retval;

  switch (field->id)
  {
  case CHAR:
  case SHORT:
  case INT:
  case FLOAT:
    return (sizeof(int));
  case DOUBLE:
    return (sizeof(double));
  case CHAR_STRING:
    retval = strlen((char *) field->data[0]) * sizeof(char) + 1;
    adjust(retval);
    return (retval);
  case ARRAY_OF_CHAR:
    retval = (field->data[1]) * sizeof(char) + sizeof(int);
    adjust(retval);
    return (retval);
  case ARRAY_OF_SHORT:
    retval = (field->data[1]) * sizeof(short) + sizeof(int);
    adjust(retval);
    return (retval);
  case ARRAY_OF_INT:
  case ARRAY_OF_FLOAT:
    return ((field->data[1]) * sizeof(int) + sizeof(int));
  case ARRAY_OF_DOUBLE:
    return ((field->data[1]) * sizeof(double) + sizeof(int));
  default:
    return (0);
  }
}

int
packed_field_bytes(field)	/* Returns the size of a field that is packed
				 * in a buffer */
  register char  *field;
{
  register int    id = *(int *) field;
  register int    arrsize, retval;

  field += sizeof(int);
  switch (id)
  {
  case CHAR:
  case SHORT:
  case INT:
  case FLOAT:
    return (sizeof(int));
  case DOUBLE:
    return (sizeof(double));
  case CHAR_STRING:
    retval = (strlen(field) * sizeof(char) + 1);
    adjust(retval);
    return (retval);
  case ARRAY_OF_CHAR:
    arrsize = *(int *) field;
    retval = arrsize * sizeof(char) + sizeof(int);
    adjust(retval);
    return (retval);
  case ARRAY_OF_SHORT:
    arrsize = *(int *) field;
    retval = arrsize * sizeof(short) + sizeof(int);
    adjust(retval);
    return (retval);
  case ARRAY_OF_INT:
  case ARRAY_OF_FLOAT:
    arrsize = *(int *) field;
    return (arrsize * sizeof(int) + sizeof(int));
  case ARRAY_OF_DOUBLE:
    arrsize = *(int *) field;
    return (arrsize * sizeof(double) + sizeof(int));
  default:
    return (0);
  }
}

char           *
skip_field(ptr)			/* skips a field that is packed in a buffer */
  register char  *ptr;
{
  register int    type = *(int *) ptr;
  register int    arrsize, retval;

  ptr += sizeof(int);
  switch (type)
  {
  case CHAR:
  case SHORT:
  case INT:
  case FLOAT:
    return (ptr + sizeof(int));
  case DOUBLE:
    return (ptr + sizeof(double));
  case CHAR_STRING:
    retval = strlen(ptr) + 1;
    adjust(retval);
    return (ptr + retval);
  case ARRAY_OF_CHAR:
    arrsize = *(int *) ptr;
    retval = (arrsize * sizeof(char) + sizeof(int));
    adjust(retval);
    return (ptr + retval);
  case ARRAY_OF_SHORT:
    arrsize = *(int *) ptr;
    retval = (arrsize * sizeof(short) + sizeof(int));
    adjust(retval);
    return (ptr + retval);
  case ARRAY_OF_INT:
  case ARRAY_OF_FLOAT:
    arrsize = *(int *) ptr;
    return (ptr + (arrsize * sizeof(int) + sizeof(int)));
  case ARRAY_OF_DOUBLE:
    arrsize = *(int *) ptr;
    return (ptr + (arrsize * sizeof(double) + sizeof(int)));
  default:
    return (ptr);
  }
}

void
rcopydata(field, ptr)		/* reverse copy from a packed buffer to
				 * variables pointed by a field struct */
  register char  *ptr;
  register FIELD *field;
{
  register int    id;

  id = actual(field->id);
  switch (id)
  {
  case CHAR:
    if (field->data[0] != NULL)
      *(char *) field->data[0] = *ptr;
    break;
  case SHORT:
    if (field->data[0] != NULL)
      *(short *) field->data[0] = *(short *) ptr;
    break;
  case INT:
    if (field->data[0] != NULL)
      *(int *) field->data[0] = *(int *) ptr;
    break;
  case FLOAT:
    if (field->data[0] != NULL)
      *(float *) field->data[0] = *(float *) ptr;
    break;
  case DOUBLE:
    if (field->data[0] != NULL)
      *((double *) field->data[0]) = *((double *) ptr);
    break;
  case CHAR_STRING:
    if (field->data[0] != NULL)
      *(char **) field->data[0] = ptr;
    break;
  case ARRAY_OF_CHAR:
    if (field->data[1] != NULL)
      *(int *) field->data[1] = *(int *) ptr;
    ptr += sizeof(int);
    if (field->data[0] != NULL)
      *(char **) field->data[0] = (char *) ptr;
    break;
  case ARRAY_OF_SHORT:
    if (field->data[1] != NULL)
      *(int *) field->data[1] = *(int *) ptr;
    ptr += sizeof(int);
    if (field->data[0] != NULL)
      *(short **) field->data[0] = (short *) ptr;
    break;
  case ARRAY_OF_INT:
    if (field->data[1] != NULL)
      *(int *) field->data[1] = *(int *) ptr;
    ptr += sizeof(int);
    if (field->data[0] != NULL)
      *(int **) field->data[0] = (int *) ptr;
    break;
  case ARRAY_OF_FLOAT:
    if (field->data[1] != NULL)
      *(int *) field->data[1] = *(int *) ptr;
    ptr += sizeof(int);
    if (field->data[0] != NULL)
      *(float **) field->data[0] = (float *) ptr;
    break;
  case ARRAY_OF_DOUBLE:
    if (field->data[1] != NULL)
      *(int *) field->data[1] = *(int *) ptr;
    ptr += sizeof(int);
    if (field->data[0] != NULL)
      *(double **) field->data[0] = (double *) ptr;
    break;
  default:
    fprintf(stderr, "RCopydata: Wrong field type\n");
  }
}

int
copydata(ptr, field)		/* copy of data from a field struct to a buffer */
  register char  *ptr;
  register FIELD *field;
{
  register int    pad, retval;

  switch (field->id)
  {
  case CHAR:
    *(int *) ptr = *(char *) field->data;
    return (sizeof(int));
  case SHORT:
    *(int *) ptr = *(short *) field->data;
    return (sizeof(int));
  case INT:
    *(int *) ptr = field->data[0];
    return (sizeof(int));
  case FLOAT:
    *(float *) ptr = *(float *) field->data;
    return (sizeof(int));
  case DOUBLE:
    *((double *) ptr) = *((double *) field->data);
    return (sizeof(double));
  case CHAR_STRING:
    strcpy(ptr, field->data[0]);
    pad = retval = strlen(ptr) + 1;
    adjust(retval);
    memset(ptr + pad, 0, retval - pad);
    return (retval);
  case ARRAY_OF_CHAR:
    *(int *) ptr = field->data[1];
    ptr += sizeof(int);
    pad = retval = field->data[1] * sizeof(char);
    copychars(ptr, field->data[0], retval);
    adjust(retval);
    memset(ptr + pad, 0, retval - pad);
    return (retval + sizeof(int));
  case ARRAY_OF_SHORT:
    *(int *) ptr = field->data[1];
    ptr += sizeof(int);
    copyshorts(ptr, field->data[0], field->data[1]);
    pad = retval = field->data[1] * sizeof(short);
    adjust(retval);
    memset(ptr + pad, 0, retval - pad);
    return (retval + sizeof(int));
  case ARRAY_OF_INT:
  case ARRAY_OF_FLOAT:
    *(int *) ptr = field->data[1];
    ptr += sizeof(int);
    copyints(ptr, field->data[0], field->data[1]);
    retval = field->data[1] * sizeof(int);
    return (retval + sizeof(int));
  case ARRAY_OF_DOUBLE:
    *(int *) ptr = field->data[1];
    ptr += sizeof(int);
    copydoubles(ptr, field->data[0], field->data[1]);
    retval = field->data[1] * sizeof(double);
    return (retval + sizeof(int));
  default:
    fprintf(stderr, "Copydata: Wrong field type\n");
    return (0);
  }
}

int
GetPort(LocalSocket, LocalAddress)	/* Get the LINDA_PORT from the local
					 * Tuple Manager. Also a good way to
					 * find out if the local Tuple Manager
					 * is up */
  int             LocalSocket;
  struct sockaddr_un *LocalAddress;
{
  int             port;
  register Rpc_Stat error;
  static struct timeval TimeOut = {10, 0};	/* 10 seconds should be enough
						 * to contact the tmanager
						 * daemon. */

  if ((error = Rpc_Call(LocalSocket, LocalAddress, (Rpc_Proc) SYSTEM_GETPORT,
			0, 0, sizeof(int), &port,
			LINDA_NRETRY_LOCAL, &TimeOut)) != RPC_SUCCESS)
  {
    fprintf(stderr, "GetPort: %s\nExiting...\n", Rpc_ErrorMessage(error));
    exit(1);
  }
  return (port);
}
