/*
 *
 *	File: prime_act.cl
 *
 *	This is a C-Linda primefinder program. It computes prime numbers        
 * upto one million.  The algorithm used is :
 *
 *		for i= lower_limit to upper_limit, stepsize = 2, do
 *                {
 *		    the number i is prime if it is not
 *                  divisible by any prime number less 
 *                  than the square root of i. 
 *        	 }
 *
 *	This algorithm is parallelized in C-Linda by dividing the range
 * to be tested for primality into chunks of suitable size (GRAIN).  E-
 * ach chunk constitutes a Linda task.  A worker grabs a task-descript-
 * or using in and immediately generates a new task-descriptor (the ne-
 * xt chunk of integers) for another worker to pick up.  It then proce-
 * eds to check integers in its own chunk for primality, report the new   
 * primes to the master using out and tries to grab another chunk.
 *	Instead of computing the square root of every number being che- 
 * cked for primality, for every new prime number added to the tuple 
 * space, its square is also added by the master.  Accordingly the con- 
 * dition of checking against the square root also transforms, as used
 * below.  A small set of first few prime numbers and their squares is 
 * defined explicitly to begin with.
 *	The number of workers desired is passed as an argument to the 
 * the program.  
 *
 *
 */   

/*
 * Initially found in a distribution of C-Linda Simulator for Sun-4, by SCA,
 * Inc.
 * 
 * Modified for POSYBL by Giannis Schoinas.
 */

#include <stdio.h>

#define NUM_INIT_PRIME  15
#define MAX_LIMIT 100000
#define MAX_GRAIN 100000 

long primes[MAX_LIMIT/10+1]={2,3, 5, 7, 11, 13, 17, 19, 23, 29, 31,  37,  41,  43,  47};
long p2[MAX_LIMIT/10+1]    ={4,9,25,49,121,169,289,361,529,841,961,1369,1681,1849,2209};


main(argc, argv)
     int argc;
     char *argv[];
{
  int		eot, first_num, i, len, num, num_primes, num_workers;
  long		new_primes[MAX_GRAIN], np2;
  int		worker(), *ptr;
  int		LIMIT, GRAIN;

  if (argc < 4) {
	printf("usage: prime_master <num_workers> <limit> grain\n");
	exit(0);
	}
  /* number of desired workers */

  num_workers = atoi(argv[1]);
  LIMIT = atoi(argv[2]);
  GRAIN = atoi(argv[3]);

  /* start the workers */

  for (i = 0; i < num_workers; ++i)
  { 
   eval_l("#2/prime_worker", NULL);
   out(lstring("init"), lint(LIMIT), lint(GRAIN));
  }

  /* number of existing primes known (as many as defined above) */

  num_primes = NUM_INIT_PRIME;

  /* the lower limit of the first chunk */
  
  first_num = primes[num_primes-1] + 2;

  /* start the Linda timer utility */

  start_timer();

  /* out an initial task (lower limit of a chunk) */
 
  out(lstring("num"), lint(first_num));

  /* collect the next set of primes reported by a worker from the tuple
     space
  */
  for (eot = 0, num = first_num; num < LIMIT; num += GRAIN) {
    in(lstring("result"), lint(num), qlnint(&ptr, &len));
    copyints(new_primes, ptr, len);


    for (i = 0; i < len; ++i, ++num_primes) {

      /* add each new prime to the existing list of primes */

      primes[num_primes] = new_primes[i];

      /* the new primes are published in the tuple space for the use
         of the workers only if their squares are less than LIMIT
      */
      if (!eot) {

 	/* compute the square of the new prime */

	np2 = new_primes[i]*new_primes[i];

 	/* if the sqaure is greater than the LIMIT then we do not 
           need any more squares (eot is set).
        */
	if (np2 > LIMIT) {
	  eot = 1;
	  np2 = -1;
	}

        /* publish each new prime and its square */
 
	out(lstring("prime"), lint(num_primes), lint(new_primes[i]), lint(np2));
      }
    }
  }

  /* inform the timer that the job is done */

  timer_split("all done.");

  /* wait until all the workers finish */

  for (i = 0; i < num_workers; ++i) in(lstring("done"));

  /* print the total number of primes less than  one million and the 
     largest prime less than one million 
  */
  printf("%d: %d\n", num_primes, primes[num_primes-1]);

  /* print the times */

  print_times();
}
