/**
 * File: cmatrix.cl
 * 
 * This is an example C-Linda program to multiply two NxN matrices. The two
 * arguments are the number of workers desired and the dimension N of the
 * matrices. The matrix multiplication is parallelised at the row level i.e.
 * each worker computes a unique row of the producvt matrix at one time.
 * 
 * The real_main routine broadly performs the following functions: 
 * 1. Initialise the two matrices. 
 * 2. Start the timers. 
 * 3. Start the workers using eval. 
 * 4. Place the matrices in the tuple space 
 * 5. Place an initial task in the tuple space using out 
 * 6. Collect the rows of the product matrix from the tuple space 
 *    using in one at a time and verify them. 
 * 7. Inform the timer of the completion of the job. 
 */

/*
 * Initially found in a distribution of C-Linda Simulator for Sun-4, by SCA,
 * Inc.
 * 
 * Modified for POSYBL by Giannis Schoinas.
 */

#include <stdio.h>

#define TRUE 1

main(argc, argv)
  int             argc;
  char          **argv;
{
  long            col[256], row[256];
  long            dim, index, true_result, *result;
  long            row_index, col_index, workers;

  if (argc != 3)
  {
    printf("Usage: %s <workers> <dim>\n", *argv);
    exit(1);
  }

  workers = atol(*++argv);
  dim = atol(*++argv);

  printf("cmatrix -- workers: %d, dim: %d\n", workers, dim);

  /* start the C-linda timer utility */

  start_timer();

  /* Initialise the two participating matrices */

  for (index = 0; index < dim; ++index)
  {
    row[index] = 3;
    col[index] = 5;
  }

  /* start workers */

  for (index = 0; index < workers; ++index)
  {
    eval_l("#2/cmatrix_worker", NULL);
  }

  /* publish the dimension N by placing in the tuple space using in */

  out(lstring("dim"), lint(dim));

  /*
   * put the the matrices in the tuple space, a row(column) at a time. One is
   * put in row major order and the other is put in column major order.
   */
  for (index = 0; index < dim; ++index)
  {
    out(lstring("row"), lint(index), lnint(row, dim));
    out(lstring("col"), lint(index), lnint(col, dim));
  }

  /* inform the timer about the completion of the setting up phase */

  timer_split("done setting up");

  /* place a row-index in the tuple space for a worker to pick */

  out(lstring("row_index"), lint(0));

  /*
   * Compute the value of each element in the result matrix. All the elements
   * are the same.
   */
  true_result = 15 * dim;

  /*
   * Retrieve each row of the product matrix, one at a time, and check whether
   * each element is correct. If incorrect, display its indices and its value.
   */
  for (index = 0; index < dim; ++index)
  {
    in(lstring("prod"), qlint(&row_index), qlnint(&result, &dim));
    for (col_index = 0; col_index < dim; ++col_index)
    {
      if (result[col_index] != true_result)
      {
	printf("got result(%ld, %ld): %ld.\n", row_index, col_index, result);
      }
    }
  }

  /*
   * Inform the Linda timer of the completion of the job and print the times.
   */
  timer_split("all done");
  print_times();

}
