
# dbs() ("debug Snow") and the other functions here serve to automate
# the process of debugging code written in Snow, meaning the portion of
# R's "parallel" library derived from the old "snow" package, e.g. the
# function clusterApply(); this part of the library will be referred to
# as "snow" below

# may also be used to debug Rdsm applications, since that library uses
# snow for launching

# requires the Unix utility "screen" (Linux, Mac, Cygwin)

# usage:

#    1.  place browser() call(s) in your worker code
#    2.  run 
#        > dbs(n)
#        where n is the desired number of snow workers 
#    3.  the function will open new windows for the workers and the manager
#    4.  you then launch your snow app from the manager window, and debug as
#        usual in the worker windows

# an outline of how the code works is given at the end of this file

# the function ws() sends a command to all worker windows;
# useful examples, called e.g. by ws('debug(f)'):

#   'debug(f)'  # debug a given function
#   'setBreakpoint("buggy.R",28)'  # set a breakpoint at a specific line
#   'c'  # resume execution of the debuggees
#   'xyz'  # print the value of the variable xyz' at all the debuggees
#   'Q'  # exit the debugger at the debuggees
#   'q()'  # exit R at the debuggees
#   'exit'  # close the debuggee windows

# for user convenience, the function dbsquitall() is provided to shut
# everything down; for each worker window, this function issues commands
# to exit the R browser (stepwise debugger), exit R, and then close the
# window (to make sure no "screen" processes remain); the function
# dbsquitrs() does part of this, if the debugger has already been quit

# see notes on screen management at the end of this file

dbs <- function(nwrkrs,app=NULL,terminal="xterm") {
   killdebugscreens()
   print("a new window will be created for your snow manager,")
   print("and one window each for your snow workers")
   readline("hit Enter ")
   # set up manager window
   makescreen("Manager",terminal)
   Sys.sleep(2)  # need throttling
   writemgrscreen("R")
   writemgrscreen("library(parallel)")
   # set up debuggee (worker) windows
   dn <- vector(mode="character",length=nwrkrs)
   for (i in 1:nwrkrs) {
      makescreen("Worker",terminal)
   }
   port <- sample(11000:11999,1)
   cmd <- 
      paste("cls <- makeCluster(",nwrkrs,",manual=T,port=",port,")",sep="")
   writemgrscreen(cmd)
   # seem to need this throttling
   Sys.sleep(2)
   readline("hit Enter ")
   rcmd <- paste("R --args MASTER=localhost PORT=",port,
      " SNOWLIB=/usr/local/lib/R/library TIMEOUT=2592000 METHODS=TRUE",sep="")
   writewrkrscreens(rcmd)
   writewrkrscreens("parallel:::.slaveRSOCK()")
   if (!is.null(app)) {
      cmd <- paste("source('",app,"')",sep="")
      writemgrscreen(cmd)
   }
   
   print("ready to launch your app in the manager window")
}

makescreen <- function(screentype,terminal) {
   dn <- screentype
   cmd <- paste(terminal,' -e "screen -S',sep="")
   cmd <- paste(cmd,dn)
   cmd <- paste(cmd,'" &')
   system(cmd)
}

# assuming all debuggees are currently in the R browser, it has them
# quit the browser, then quit R, then close the window
dbsquitall <- function() {
   writewrkrscreens("Q")
   dbsquitrs()
}

# quit R, then close the window
dbsquitrs <- function() {
   writewrkrscreens("q()")
   writewrkrscreens("n")
   writewrkrscreens("exit")
}

writemgrscreen <- function(cmd) {
   cmd <- paste(cmd,'\n',sep="")
   cmd <- shQuote(cmd)
   mgr <- getmgrscreen()
   tosend <- paste('screen -S ',mgr,' -X stuff ', cmd, sep="")
   system(tosend)
}

writewrkrscreens <- function(cmd) {
   cmd <- paste(cmd,'\n',sep="")
   for (dn in getwrkrscreens()) {
      tosend <- paste('screen -S ',dn,' -X stuff ', "'",cmd,"'", sep="")
      system(tosend)
   }
}

getwrkrscreens <- function() {
   scrns <- vector()
   tmp <- system("screen -ls",intern=T)
   for (l in tmp) {
      # debuggee screen?
      if (length(grep("Worker",l)) > 0) {
         scrn <- getscreenname("Worker",l)
         scrns <- c(scrns,scrn)
      }
   }
   scrns
}

getmgrscreen <- function(warning=T) {
   tmp <- system("screen -ls",intern=T)
   for (l in tmp) {
      # mgr screen?
      if (length(grep("Manager",l)) > 0) {
         scrn <- getscreenname("Manager",l)
         return(scrn)
      }
   }
   if (warning) print("warning:  no manager screen found")
   return(NULL)
}

# for a line lsentry from the output of "screen -ls" and given
# screentype (here "Worker" or "Manager"), returns the screen name, in
# the form of OS process number + type
getscreenname <- function(screentype,lsentry) {
   # remove leading TAB character
   substr(lsentry,1,1) <- ""
   startoftype <- gregexpr(screentype,lsentry)[[1]]
   endofname <- startoftype + nchar(screentype) - 1
   substr(lsentry,1,endofname)
}

killdebugscreens <- function() {
   scrns <- getwrkrscreens()
   scrns <- c(scrns,getmgrscreen(warning=F))
   for (scrn in scrns) {
      cmd <- paste("screen -X -S ",scrn," kill",sep="")
      system(cmd)
   }
}

# abbreviations
ws <- writewrkrscreens
ks <- killdebugscreens

# screen management:

#    the routines here, e.g. killdebugscreens() should be sufficient to
#    delete zombie screens and so on, but it is worth knowing these
#    commands, if at least to understand the above code:

#        screen -ls:  lists all screens, in a OS process number + name
#                     format
#        screen -S screenname -X stuff cmd:  write cmd to the specified screen
#        screen -X -S screenname kill:  kill the specified screen

# dbs() does the following:

#   uses "screen" create new terminal windows in which the debuggees will run,
#      one window for each worker, and one for the manager
#   sends a makeCluster() command to the manager window
#   starts up R processes in the worker windows, each communicating 
#      with the manager
#   writes the string "parallel:::.slaveRSOCK()" to each of the windows,
#      which means they loop around looking for commands from the manager,
#      generated by the "snow" code to be debugged, e.g. from clusterApply()

